/*
Help.tree and Help.gui - a scheme to allow UGens, no wait I mean ALL classes, 
to be "self-classified" and provide a pleasant-ish browsing interface. No wait, 
let's put all help docs into the tree too! Yeah!

By Dan Stowell, 2007
with lots of input from Scott Wilson
and crossplatform tips from nescivi

Try it:
Help.gui

Help.dumpTree

see also:
Class.browse
*/

Help {
	classvar tree, categoriesSkipThese, fileslist;
	classvar <filterUserDirEntries;
	
	*initClass {
		categoriesSkipThese = [Filter, BufInfoUGenBase, InfoUGenBase, MulAdd, BinaryOpUGen, 
						UnaryOpUGen, BasicOpUGen, LagControl, TrigControl, MultiOutUGen, ChaosGen,
			Control, OutputProxy, AbstractOut, AbstractIn, Object, Class];
		filterUserDirEntries = [ "Extensions", "SuperCollider", "SuperCollider3", "Help", "svn", "share", "classes", "trunk", "Downloads" ];
	}
	
	*tree { |sysext=true,userext=true|
		var classes, node, subc, helpRootLen;
		var helpExtensions = ['html', 'scd', 'rtf', 'rtfd'];
		var helpDirs = Array.new;
		var thisHelpExt;
		if(tree.isNil, {
			// Building the tree - base class was originally UGen
			
			// Help file paths - will be used for categorising, if categories is nil or if not a class's helpfile.
			// Otherwise they'll be stored for quick access to helpfile.
			fileslist = IdentityDictionary.new;
			helpDirs = helpDirs.add( Platform.helpDir );
			if ( sysext ,{
				helpDirs = helpDirs.add( Platform.systemExtensionDir );
			});
			if ( userext ,{
				helpDirs = helpDirs.add( Platform.userExtensionDir );
			});
			
			// Now check each class's ".categories" response
			classes = Object.allSubclasses.difference(categoriesSkipThese).reject({|c| c.asString.beginsWith("Meta_")});
			tree = Dictionary.new(8);
			classes.do({|class| this.addCatsToTree(class, fileslist)});
			
			// Now add the remaining ones to the tree - they're everything except the classes which 
	//      have declared their own categorisation(s).
			
			helpDirs.do{ |helpDir|
				this.addDirTree( helpDir,tree );
			};
		});
		^tree;
	}

	*addUserFilter{ |subpath|
		filterUserDirEntries = filterUserDirEntries.add( subpath );
		this.forgetTree;
	}

	*addDirTree{ |helppath,tree|
		var helpExtensions = ['html', 'scd', 'rtf', 'rtfd'];
		var subfileslist;
		var node, subc, helpRootLen, thisHelpExt;

		subfileslist = IdentityDictionary.new;

		PathName.new(helppath.standardizePath).filesDo({|pathname|
				if( helpExtensions.includes(pathname.extension.asSymbol)
					&& pathname.fullPath.contains("3vs2").not
					&& pathname.fullPath.contains("help-scripts").not
					, {
						subfileslist[pathname.fileNameWithoutDoubleExtension.asSymbol] = pathname.fullPath;
						fileslist[pathname.fileNameWithoutDoubleExtension.asSymbol] = pathname.fullPath;
					})
			});

		helpRootLen = (helppath.standardizePath).size + 1;
		subfileslist.keysValuesDo({ |classsym, path|

			if ( helppath == Platform.helpDir,
				{
					subc = path[helpRootLen..].split($/);
					subc = subc[0..subc.size-2]; // Ignore "Help" and the filename at the end
				},{
					//helpRootLen = "~".standardizePath;
					if ( helppath == Platform.systemExtensionDir,
						{
							subc = path[helpRootLen..].split($/);
							subc = [ "SystemExtensions" ] ++ subc;
							//subc.postcs;
						});
					if ( helppath == Platform.userExtensionDir,
						{
							helpRootLen = "~/".absolutePath.size; // + 1;
							subc = path[helpRootLen..].split($/);
							subc = [ "UserExtensions" ] ++ subc;
							// check for common superfluous names that may confuse the categorisation;
							filterUserDirEntries.do{ |spath|
								subc = subc.reject{ |it| 
									it == spath;
								};
							};
							// check for double entries (e.g. SwingOSC)
							subc[..subc.size-2].do{ |it,i|
								var subset;
								subset = subc[..i-1];
								if ( subset.detect( { |jt| jt == it } ).size > 0, {
									subc = subc[..i-1] ++ subc[i+1..];
								});
							};
						});
					subc = subc[..subc.size-2];
				}
			);
			thisHelpExt = helpExtensions.select{ |ext|
				subc.last.endsWith("."++ext)
			};
			if ( thisHelpExt.size > 0 , {
				subc = subc[..subc.size-2];
			});
			
			subc = subc.collect({|i| "[["++i++"]]"});
			node = tree;
			// Crawl up the tree, creating hierarchy as needed
			subc.do({|catname|
				if(node[catname].isNil, {
					node[catname] = Dictionary.new(3);
				});
				node = node[catname];
			});
			// "node" should now be the tiniest branch
			node[classsym.asClass ? classsym] = path;
		});
	}
	
	*forgetTree {
		tree = nil;
	}
	
	*dumpTree { |node, prefix=""|
		node = node ?? {this.tree};
		node.keysValuesDo({ |key, val|
			if(val.isKindOf(Dictionary), {
				(prefix + key).postln;
				this.dumpTree(val, prefix ++ "   ");
			}, {
				(prefix + key ++":" + val).postln;
			});
		});
	}
	
	*addCatsToTree { |class, fileslist|
		var subc, node;
		
		if(class.categories.isNil.not, {
			class.categories.do({|cat|
				subc = cat.split($>).collect({|i| "[["++i++"]]"});
				node = tree;
				// Crawl up the tree, creating hierarchy as needed
				subc.do({|catname|
					if(node[catname].isNil, {
						node[catname] = Dictionary.new(3);
					});
					node = node[catname];
				});
				// "node" should now be the tiniest branch
				node[class] = fileslist[class.asSymbol] ? "";
			});
			
			// Class has been added to list so we're OK
			fileslist.removeAt(class.asSymbol);
		}); // end if
		
	}


*gui {

	// called from Help menu
	"open http://www.ixi-audio.net".unixCmd;

} 


	*all {
		//		^this.new("Help/").dumpToDoc("all-helpfiles");
		var doc;
		var helpExtensions = ['html', 'scd', 'rtf', 'rtfd'];
		var str = CollStream.new;
		doc = Document.new("all-helpfiles");
		[       Platform.helpDir,
			Platform.systemExtensionDir,
			Platform.userExtensionDir
		].do{ |it|
			PathName.new( it ).foldersWithoutSVN.do{ |folderPn|
				str << folderPn.fullPath << Char.nl;
				folderPn.filesDo { |filePn|
					if 
					(helpExtensions.includes(filePn.extension.asSymbol)) {
						str << Char.tab << 
						filePn.fileNameWithoutExtension  << Char.nl;
					}
				};
			}
		};
		doc.string = str.collection;
	}
	
	// Iterates the tree, finding the help-doc paths and calling action.value(docname, path)
	*do { |action|
		this.pr_do(action, this.tree);
	}
	*pr_do { |action, curdict|
		curdict.keysValuesDo{|key, val|
			if(val.class == Dictionary){
				this.pr_do(action, val) // recurse
			}{
				action.value(key.asString, val)
			}
		}
	}
	
	/*
	Help.searchGUI
	*/
	*searchGUI {
		var win, qbox, resultsview, results, winwidth=600, statictextloc;
		
		win = GUI.window.new("<< Search SC Help >>", Rect(100, 400, winwidth, 600));
		
		statictextloc = Rect(10, 10, winwidth-20, 200);
		
		// SCTextField
		qbox = GUI.textField.new(win, Rect(0, 0, winwidth, 50).insetBy(50,15))
			.resize_(2)
			.action_{ |widget|
				resultsview.removeAll;
				if(widget.value != ""){
					results = this.search(widget.value);
					// Now add the results!
					if(results.size == 0){
						GUI.staticText.new(resultsview, statictextloc)
							.resize_(5)
							.string_("No results found.");
					}{
						results.do{|res, index|
							res.drawRow(resultsview, Rect(0, index*30, winwidth, 30));
						}
					};
				};
			};
		
		/*
		// dividing line? hmm, double-drawing issue on my mac
		win.drawHook_{
			Pen.color = Color.black;
			Pen.moveTo(Point(0, 49));
			Pen.lineTo(Point(winwidth, 49));
			Pen.stroke;
		};
		*/
		
		// SCScrollView
		resultsview = GUI.scrollView.new(win, Rect(0, 50, winwidth, 550))
				.resize_(5);
		
		GUI.staticText.new(resultsview, statictextloc)
			.resize_(5)
			.string_("Type a word above and press enter.\nResults will appear here.");
		
		win.front;
		qbox.focus;
		^win;
	}
	
	// Returns an array of hits as HelpSearchResult instances
	*search { |query, ignoreCase=true|
		var results = List.new, file, ext, docstr, pos;
		this.do{ |docname, path|
			if(path != ""){	
				if(docname.find(query, ignoreCase).notNil){
					results.add(HelpSearchResult(docname, path, 100 / (docname.size - query.size + 1), ""));
				}{
					ext = path.splitext[1];
					// OK, let's open the document, see if it contains the string... HEAVY!
					file = File(path, "r");
					if(file.isOpen){
						docstr = ext.switch(
							"html", {file.readAllStringHTML},
							"htm",  {file.readAllStringHTML},
							"rtf",  {file.readAllStringRTF},
							        {file.readAllString}
							);
						file.close;
						pos = docstr.findAll(query, ignoreCase);
						if(pos.notNil){
							results.add(HelpSearchResult(docname, path, pos.size, docstr[pos[0] ..  pos[0]+50]));
						}
					}{
						"File:isOpen failure: %".format(path).postln;
					}
				}
			}{
				//"empty path: %".format(docname).postln;
			}
		};
		results = results.sort;
		
		^results
	}

} // End class


HelpSearchResult {
	var <>docname, <>path, <>goodness, <>context;
	*new{|docname, path, goodness, context|
		^this.newCopyArgs(docname, path, goodness, context);
	}
	
	asString {
		^ "HelpSearchResult(%, %, %, %)".format(docname, path.basename, goodness, this.contextTrimmed)
	}
	// used for sorting:
	<= { |that|
		^ this.goodness >= that.goodness
	}
	
	contextTrimmed {
		^context.tr($\n, $ ).tr($\t, $ )
	}
	
	drawRow { |parent, bounds|
		// SCButton
		GUI.button.new(parent, bounds.copy.setExtent(bounds.width * 0.3, bounds.height).insetBy(5, 5))
				.states_([[docname]]).action_{ path.openHTMLFile };
		
		GUI.staticText.new(parent, bounds.copy.setExtent(bounds.width * 0.7, bounds.height)
										.moveBy(bounds.width * 0.3, 0)
										.insetBy(5, 5))
				.string_(this.contextTrimmed);
		
	}
}


+ Object {

// Classes may override this to specify where they fit in a thematic classification,
// if they want to classify differently than the help doc location would indicate.
//
// Multiple categorisations are allowed (hence the array).
//
// Extension libs (which won't automatically get allocated, since their help won't be in the main
//   help tree) SHOULD override this to specify where best to fit.
//   (Note: *Please* use the "Libraries" and/or "UGens" main classifications, those are the best
//   places for users to find extensions docs. Don't add new "root" help categories, that's 
//   not good for GUI usability.)
//
// Each categorisation should be a string using ">" marks to separate categories from subcategories.
// For examples see (e.g.) SinOsc, Gendy1, LPF, Integrator, EnvGen
//*categories {	^ #["Unclassified"]	}
*categories {	^ nil	}

}

+ Pattern {
	*categories {	^ #["Streams-Patterns-Events>Patterns"] }
}

// This allows it to be called from sclang menu item
+ Process {
	helpGui {
		Help.gui
	}
}
